create or replace type tp_instr force authid current_user as object
(
  curnum  int,
  howmany int,
  static function get_len(ia sys.odciindexinfo) return int,
  static function odcigetinterfaces(ifclist out sys.odciobjectlist)
    return number,
  static function odciindexcreate(ia    sys.odciindexinfo,
                                  parms varchar2,
                                  env   sys.odcienv) return number,
  static function odciindextruncate(ia sys.odciindexinfo, env sys.odcienv)
    return number,
  static function odciindexdrop(ia sys.odciindexinfo, env sys.odcienv)
    return number,
  static function odciindexinsert(ia     sys.odciindexinfo,
                                  rid    rowid,
                                  newval varchar2,
                                  env    sys.odcienv) return number,
  static function odciindexdelete(ia     sys.odciindexinfo,
                                  rid    rowid,
                                  oldval varchar2,
                                  env    sys.odcienv) return number,
  static function odciindexupdate(ia     sys.odciindexinfo,
                                  rid    rowid,
                                  oldval varchar2,
                                  newval varchar2,
                                  env    sys.odcienv) return number,
  static function odciindexstart(sctx  in out tp_instr,
                                 ia    sys.odciindexinfo,
                                 op    sys.odcipredinfo,
                                 qi    sys.odciqueryinfo,
                                 strt  number,
                                 stop  number,
                                 value varchar2,
                                 env   sys.odcienv) return number,
  member function odciindexfetch(self  in out tp_instr,
                                 nrows number,
                                 rids  out sys.odciridlist,
                                 env   sys.odcienv) return number,
  member function odciindexclose(env sys.odcienv) return number
);
/

create or replace type body tp_instr is

  static function get_len(ia sys.odciindexinfo) return int as
    stmt   varchar2(2000);
    result int;
  begin
    stmt := 'select to_number(parameters default 0 on conversion error)
               from all_indexes where owner = ''{idx_schema}'' and index_name = ''{idx}''';
    stmt := replace(stmt, '{idx_schema}', ia.indexschema);
    stmt := replace(stmt, '{idx}', ia.indexname);
    execute immediate stmt into result;
    return case when result between 3 and 10 then result else 3 end;
  end get_len;

  /*
    Unused parameters cannot be removed from ODCI routines
    because the are the part of interface.
    For example, oldval in odciindexdelete or env in all routines
  */

  static function odcigetinterfaces(ifclist out sys.odciobjectlist)
    return number is
  begin
    ifclist := sys.odciobjectlist(sys.odciobject('SYS', 'ODCIINDEX2'));
    return odciconst.success;
  end odcigetinterfaces;

  static function odciindexcreate(ia    sys.odciindexinfo,
                                  parms varchar2,
                                  env   sys.odcienv) return number is
    stmt varchar2(2000);
    l    int;
  begin
    l    := to_number(parms default 0 on conversion error);
    l    := case when l between 3 and 10 then l else 3 end;
    stmt := 'create table {idx_schema}.{idx}$
            (sub_str, row_id, idx, constraint pk_{idx}$ primary key (sub_str, row_id, idx))
            organization index parallel {dop}
            as
            select sub_str, t.rowid row_id, idx
              from {tbl_schema}.{tbl} t,
                   lateral (select lower(substr({col}, level, {l})) sub_str, level idx
                              from dual
                            connect by level <= length({col}))';
    stmt := replace(stmt, '{idx_schema}', ia.indexschema);
    stmt := replace(stmt, '{idx}', ia.indexname);
    stmt := replace(stmt, '{tbl_schema}', ia.indexcols(1).tableschema);
    stmt := replace(stmt, '{tbl}', ia.indexcols(1).tablename);
    stmt := replace(stmt, '{col}', ia.indexcols(1).colname);
    stmt := replace(stmt, '{dop}', ia.indexparadegree);
    stmt := replace(stmt, '{l}', l);
    execute immediate stmt;
    stmt := 'alter table {idx_schema}.{idx}$ noparallel';
    stmt := replace(stmt, '{idx_schema}', ia.indexschema);
    stmt := replace(stmt, '{idx}', ia.indexname);
    execute immediate stmt;
    return odciconst.success;
  end;

  static function odciindexdrop(ia sys.odciindexinfo, env sys.odcienv)
    return number is
    stmt varchar2(2000);
  begin
    stmt := 'drop table ' || ia.indexschema || '.' || ia.indexname || '$';
    execute immediate stmt;
    return odciconst.success;
  end;

  static function odciindextruncate(ia sys.odciindexinfo, env sys.odcienv)
    return number is
    stmt varchar2(2000);
  begin
    stmt := 'truncate table ' || ia.indexschema || '.' || ia.indexname || '$';
    execute immediate stmt;
    return odciconst.success;
  end;

  static function odciindexinsert(ia     sys.odciindexinfo,
                                  rid    rowid,
                                  newval varchar2,
                                  env    sys.odcienv) return number is
    stmt varchar2(2000);
  begin
    stmt := 'insert into {idx_schema}.{idx}$
             select lower(substr(''{new_val}'', level, {l})) sub_str, ''{rid}'', level idx
               from dual
             connect by level <= length(''{new_val}'')';
    stmt := replace(stmt, '{idx_schema}', ia.indexschema);
    stmt := replace(stmt, '{idx}', ia.indexname);
    stmt := replace(stmt, '{rid}', rid);
    stmt := replace(stmt, '{new_val}', newval);
    stmt := replace(stmt, '{l}', get_len(ia));
    execute immediate stmt;
    return odciconst.success;
  end;

  static function odciindexdelete(ia     sys.odciindexinfo,
                                  rid    rowid,
                                  oldval varchar2,
                                  env    sys.odcienv) return number is
    stmt varchar2(2000);
  begin
    stmt := 'delete from {idx_schema}.{idx}$ where row_id = ''{rid}''';
    stmt := replace(stmt, '{idx_schema}', ia.indexschema);
    stmt := replace(stmt, '{idx}', ia.indexname);
    stmt := replace(stmt, '{rid}', rid);
    execute immediate stmt;
    return odciconst.success;
  end;

  static function odciindexupdate(ia     sys.odciindexinfo,
                                  rid    rowid,
                                  oldval varchar2,
                                  newval varchar2,
                                  env    sys.odcienv) return number is
  begin
    -- NoFormat Start
    return odciindexdelete(ia, rid, oldval, env) + odciindexinsert(ia, rid, newval, env);
    -- NoFormat End 
  end;

  static function odciindexstart(sctx  in out tp_instr,
                                 ia    sys.odciindexinfo,
                                 op    sys.odcipredinfo,
                                 qi    sys.odciqueryinfo,
                                 strt  number,
                                 stop  number,
                                 value varchar2,
                                 env   sys.odcienv) return number is
    stmt varchar2(2000);
    cnum integer;
  begin
    if not (bitand(op.flags, odciconst.predexactmatch) =
        odciconst.predexactmatch and strt = 1) then
      raise_application_error(-20001,
                              'Predicate can be only in a form "operator(column, value) = 1"');
    end if;
    stmt := 'select t.rowid
               from {tbl_schema}.{tbl} t
              where t.rowid in (select i.row_id from {idx_schema}.{idx}$ i
                                 where i.sub_str like lower(substr(''{value}'', 1, {l})) || ''%'')
                 -- this condition is required because length(value) may exceed token length in auxiliary table
                and lower(t.{col}) like ''%{value}%''';
    stmt := replace(stmt, '{idx_schema}', ia.indexschema);
    stmt := replace(stmt, '{idx}', ia.indexname);
    stmt := replace(stmt, '{tbl_schema}', ia.indexcols(1).tableschema);
    stmt := replace(stmt, '{tbl}', ia.indexcols(1).tablename);
    stmt := replace(stmt, '{col}', ia.indexcols(1).colname);
    stmt := replace(stmt, '{value}', value);
    stmt := replace(stmt, '{l}', get_len(ia));
    cnum := dbms_sql.open_cursor;
    dbms_sql.parse(cnum, stmt, dbms_sql.native);
    -- set context as the cursor number
    sctx := tp_instr(cnum, 0);
    return odciconst.success;
  end;

  member function odciindexfetch(self  in out tp_instr,
                                 nrows number,
                                 rids  out sys.odciridlist,
                                 env   sys.odcienv) return number is
    cnum    integer;
    rid_tab dbms_sql.varchar2_table;
    rlist   sys.odciridlist := sys.odciridlist();
    i       integer;
    d       integer;
  begin
    cnum := self.curnum;
    if self.howmany = 0 then
      dbms_sql.define_array(cnum, 1, rid_tab, nrows, 1);
      d := dbms_sql.execute(cnum);
    end if;
    d := dbms_sql.fetch_rows(cnum);
    -- null value in the end marks end of the fetch
    if d = nrows then
      rlist.extend(d);
    else
      rlist.extend(d + 1);
    end if;
    dbms_sql.column_value(cnum, 1, rid_tab);
    for i in 1 .. d loop
      rlist(i) := rid_tab(i + self.howmany);
    end loop;
    self.howmany := self.howmany + d;
    rids         := rlist;
    return odciconst.success;
  end;

  member function odciindexclose(env sys.odcienv) return number is
    cnum integer;
  begin
    cnum := self.curnum;
    dbms_sql.close_cursor(cnum);
    return odciconst.success;
  end;
end;
/